/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.vha.vuid.rest.testng;

import static gov.vha.vuid.rest.constants.Constants.VUID_DATA_STORE_ROOT_LOCATION_PROPERTY;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.Properties;

import javax.ws.rs.client.Entity;
import javax.ws.rs.core.Application;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import org.glassfish.grizzly.http.util.Header;
import org.glassfish.jersey.test.JerseyTestNg;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.rest.api.data.vuid.RestVuidBlockData;
import gov.vha.vuid.rest.ApplicationConfig;
import gov.vha.vuid.rest.VuidLocalJettyRunner;
import gov.vha.vuid.rest.api1.RestPaths;
import gov.vha.vuid.rest.data.VuidService;
import gov.vha.vuid.rest.data.VuidServiceImpl;
import gov.vha.vuid.rest.session.RequestParameters;

/**
 * {@link RestTest}
 * Testing framework for doing full cycle testing - this launches the REST server in a grizzly container, and makes REST requests via a loop
 * back call.
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 */
public class RestTest extends JerseyTestNg.ContainerPerClassTest
{
	// Example tokens
	//	private static final String readOnlyToken="%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC6%5CxF2%5CxE8%5CxA5%5CxD8%5CxE3t%5CxFFUK%22%2C+%22%2CJ%5Cx83%5CxA3%5Cx13k%5Cx96%5CxFC%5CxE6%5CxF3%5CxCF%5CxF2%7C%5CxB8MK%22%2C+%224%5Cf%5Cx94%5CxB0%5Ce%7C%5Cx9C%5CxB0%5CxA6%5CxA8%5CxE1%5CxE1t%5CxBC%5CvK%22%2C+%22a%40%5Cx8A%5CxACT%7B%5Cx9C%5CxB3%5CxE8%5CxAC%5CxA7%5Cx95%5Cx17%5CxDBiL%22%5D";
	//	private static final String gregToken="%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC2%5CxEE%5CxFA%5CxE1%5Cx91%5CxBF3%5CxA9%5Cx16K%22%2C+%22%7EK%5CxC4%5CxEFX%7C%5Cx96%5CxA8%5CxA3%5CxA2%5CxC4%5CxB1%3D%5CxFF%5Cx01K%22%2C+%22oC%5Cx83%5CxF7%40%3A%5Cx94%5CxAC%5CxAF%5CxB6%5CxE1%5CxF4c%5CxB8%5CbK%22%2C+%22+M%5Cx89%5CxB8Xe%5CxF9%5CxD4%5CxC0%5CxDB%5CxAB%5Cx99%5Ce%5CxD7e%40%22%5D";
	//	private static final String joelToken="%5B%22u%5Cf%5Cx8F%5CxB1X%5C%22%5CxC7%5CxF2%5CxE8%5CxA5%5CxD8%5CxE3t%5CxFFUK%22%2C+%22%2CJ%5Cx83%5CxA3%5Cx13k%5Cx96%5CxFC%5CxE6%5CxF3%5CxCF%5CxF2%7C%5CxB8MK%22%2C+%224%5Cf%5Cx8C%5CxBA%5Cx1Ft%5CxDD%5CxB5%5CxA4%5CxB8%5CxC0%5CxE9Q%5CxAB%5CnK%22%2C+%22z%5D%5Cx83%5CxAFT%7B%5Cx9C%5CxB3%5CxE8%5CxAC%5CxA7%5Cx95%5Cx17%5CxDBiL%22%5D";

	private static final String TEST_SSO_TOKEN = "TestUser:super_user,editor,read_only,approver,administrator,reviewer,manager,vuid_requestor";
	private static final String TEST_READ_ONLY_SSO_TOKEN = "TestReadOnlyUser:read_only";

	@Override
	protected Application configure()
	{
		try
		{
			System.out.println("Launching Jersey within Grizzley for tests");
			new File("target/test.data").mkdirs();
			System.setProperty(VUID_DATA_STORE_ROOT_LOCATION_PROPERTY, "target/test.data");
			return VuidLocalJettyRunner.configureJerseyServer();
		}
		catch (Exception e)
		{
			throw new RuntimeException(e);
		}
	}

	@BeforeClass
	public void testDataLoad()
	{
		//Load in the test data
		try
		{
			System.out.println("Awaiting service startup...");
			int attempt = 0;
			while (!ApplicationConfig.getInstance().isServiceReady())
			{
				System.out.print(".");
				attempt++;
				if (attempt % 50 == 0) {
					System.out.println();
				}
				Thread.sleep(50);
			}
			System.out.println("System started");
		}
		catch (Exception e)
		{
			Assert.fail("Startup failed", e);
		}
	}

	private String expectFail(Response response)
	{
		String temp = response.readEntity(String.class);
		if (response.getStatus() == Status.OK.getStatusCode())
		{
			Assert.fail("Should have failed but did not. Response code " + response.getStatus() + " - " + Status.fromStatusCode(response.getStatus()) + temp);
		}

		return temp;
	}

	private Response checkFail(Response response)
	{
		return assertResponseStatus(response, Status.OK.getStatusCode());
	}
	private Response assertResponseStatus(Response response, int expectedStatus)
	{
		if (response.getStatus() != expectedStatus)
		{
			Assert.fail("Unexpected response code " + response.getStatus() + " \"" + Status.fromStatusCode(response.getStatus()) + "\". Expected " + expectedStatus + " \"" + Status.fromStatusCode(expectedStatus) + "\". "
					+ " " + response.readEntity(String.class));
		}
		return response;
	}

//	@Test
//	public void testMe2() throws JsonParseException, JsonMappingException, IOException
//	{		
//		final int blockSize = 10;
//		Response getResponse = target(RestPaths.writePathComponent + RestPaths.vuidAPIsPathComponent + RestPaths.allocateTestComponent)
//				.queryParam(RequestParameters.blockSize, blockSize)
//				.queryParam(RequestParameters.reason, "A test reason")
//				.queryParam(RequestParameters.ssoToken, TEST_SSO_TOKEN)
//				.request()
//				.header(Header.Accept.toString(), MediaType.APPLICATION_JSON).get();
//		String getResponseResult = checkFail(getResponse).readEntity(String.class);
//		RestVuidBlockData vuids = new ObjectMapper().readValue(getResponseResult, RestVuidBlockData.class);
//		
//		Assert.assertNotNull(vuids);
//
//		Assert.assertTrue(vuids.startInclusive == -1); // TODO change when method unstubbed
//		Assert.assertTrue(vuids.endInclusive == -blockSize); // TODO change when method unstubbed
//	}

	@Test
	public void testMe1() throws JsonParseException, JsonMappingException, IOException
	{		
		final int blockSize = 10;
		final String reason = "A test reason";

		Response getResponse = target(RestPaths.writePathComponent + RestPaths.vuidAPIsPathComponent + RestPaths.allocateComponent)
				.queryParam(RequestParameters.ssoToken, TEST_SSO_TOKEN)
				.queryParam(RequestParameters.blockSize, blockSize)
				.queryParam(RequestParameters.reason, reason)
				.request()
				.header(Header.Accept.toString(), MediaType.APPLICATION_JSON).post(Entity.xml(""));
		String getResponseResult = checkFail(getResponse).readEntity(String.class);
		RestVuidBlockData vuids = new ObjectMapper().readValue(getResponseResult, RestVuidBlockData.class);
		
		Assert.assertNotNull(vuids);
		// This shouldn't be necessary as a test will use a fresh DB 
		//Assert.assertTrue((vuids.startInclusive > 0 && vuids.endInclusive > 0) || (vuids.startInclusive < 0 && vuids.endInclusive < 0));
		Assert.assertEquals(vuids.startInclusive, -1);
		Assert.assertEquals(vuids.endInclusive, -10);
		Assert.assertEquals(Math.abs(Math.abs(vuids.endInclusive) - Math.abs(vuids.startInclusive)), blockSize - 1);
	}
	
	@Test
	public void test_overMaxBlockSize()
			throws JsonParseException, JsonMappingException, IOException
	{
		final int blockSize = 1000001;
		final String reason = "A test reason 1000001";

		Response getResponse = target(RestPaths.writePathComponent + RestPaths.vuidAPIsPathComponent + RestPaths.allocateComponent)
				.queryParam(RequestParameters.ssoToken, TEST_SSO_TOKEN)
				.queryParam(RequestParameters.blockSize, blockSize)
				.queryParam(RequestParameters.reason, reason)
				.request()
				.header(Header.Accept.toString(), MediaType.APPLICATION_JSON).post(Entity.xml(""));
		
		// Throws RestException before we get the response
	}
	
	@Test
	public void test_negativeBlockSize()
			throws JsonParseException, JsonMappingException, IOException
	{
		final int blockSize = -1;
		final String reason = "A test reason -1";

		Response getResponse = target(RestPaths.writePathComponent + RestPaths.vuidAPIsPathComponent + RestPaths.allocateComponent)
				.queryParam(RequestParameters.ssoToken, TEST_SSO_TOKEN)
				.queryParam(RequestParameters.blockSize, blockSize)
				.queryParam(RequestParameters.reason, reason)
				.request()
				.header(Header.Accept.toString(), MediaType.APPLICATION_JSON).post(Entity.xml(""));
		Assert.assertEquals(getResponse.getStatus(), Status.BAD_REQUEST.getStatusCode());
	}
	
	@Test
	public void test_zeroBlockSize()
			throws JsonParseException, JsonMappingException, IOException
	{
		final int blockSize = 0;
		final String reason = "A test reason 0";

		Response getResponse = target(RestPaths.writePathComponent + RestPaths.vuidAPIsPathComponent + RestPaths.allocateComponent)
				.queryParam(RequestParameters.ssoToken, TEST_SSO_TOKEN)
				.queryParam(RequestParameters.blockSize, blockSize)
				.queryParam(RequestParameters.reason, reason)
				.request()
				.header(Header.Accept.toString(), MediaType.APPLICATION_JSON).post(Entity.xml(""));
		Assert.assertEquals(getResponse.getStatus(), Status.BAD_REQUEST.getStatusCode());
	}
	
	@Test
	public void test_isReady()
	{
		Assert.assertTrue(LookupService.getService(VuidService.class).isReady());
	}
	
	@Test
	public void test_isProductionMode()
	{
		Assert.assertFalse(LookupService.getService(VuidService.class).isProductionMode());
	}
	
	@Test
	public void test_loadPropertiesInWindow() 
			throws NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException, FileNotFoundException, IOException
	{
		Properties p_tmp = new Properties();
		File file = new File(getClass().getClassLoader().getResource("database_vuid.yml").getFile());
		p_tmp.load(new FileReader(file));
		VuidServiceImpl vs = LookupService.getService(VuidServiceImpl.class);
		Method loadPropertiesInWindow = VuidServiceImpl.class
					.getDeclaredMethod("loadPropertiesInWindow", File.class);
		loadPropertiesInWindow.setAccessible(true);
		Properties p = (Properties) loadPropertiesInWindow.invoke(vs, file);
		Assert.assertEquals(p, p_tmp);
	}
	
	
	@Test
	public void test_getLogEventsTargetURL() throws Exception
	{
		File file = new File(getClass().getClassLoader().getResource("database_vuid.yml").getFile());
		String logUrl = "https://MSI/rails_prisme/log_event?security_token=%255B%2522u%255Cf%255Cx92%255CxBC%255Cx17%257D%255CxD1%255CxE4%255CxFB%255CxE5%255Cx98%255CxA6%2525%255CxE8VK%2522%252C%2B%2522%253F%255Cx17%255CxD2%255CxA8v%255Cx14%255CxFF%255CxD2%255CxC6%255CxDD%255CxAD%255Cx9F%255Cx1D%255CxD1cF%2522%255D";
		Properties p_tmp = new Properties();
		p_tmp.load(new FileReader(file));
		VuidServiceImpl vs = LookupService.getService(VuidServiceImpl.class);
		Field props = VuidServiceImpl.class.getDeclaredField("PROPS");
		props.setAccessible(true);
		props.set(vs, p_tmp);
		@SuppressWarnings("unchecked")
		String gotLogUrl = vs.getLogEventsTargetURL().orElse("Failed");
		Assert.assertEquals(gotLogUrl, logUrl);
	}
}